'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

/**
 * Copyright (c) 2013-present, Facebook, Inc.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */

/**
 * Use invariant() to assert state which your program assumes to be true.
 *
 * Provide sprintf-style format (only %s is supported) and arguments
 * to provide information about what broke and what you were
 * expecting.
 *
 * The invariant message will be stripped in production, but the invariant
 * will remain to ensure logic does not differ in production.
 */

var NODE_ENV = process.env.NODE_ENV;

var invariant = function(condition, format, a, b, c, d, e, f) {
  if (NODE_ENV !== 'production') {
    if (format === undefined) {
      throw new Error('invariant requires an error message argument');
    }
  }

  if (!condition) {
    var error;
    if (format === undefined) {
      error = new Error(
        'Minified exception occurred; use the non-minified dev environment ' +
        'for the full error message and additional helpful warnings.'
      );
    } else {
      var args = [a, b, c, d, e, f];
      var argIndex = 0;
      error = new Error(
        format.replace(/%s/g, function() { return args[argIndex++]; })
      );
      error.name = 'Invariant Violation';
    }

    error.framesToPop = 1; // we don't care about invariant's own frame
    throw error;
  }
};

var invariant_1 = invariant;

var defaultWarnings = {
    missing: function (propName, propValue, componentName) {
        return "Prop `" + propName + "` marked as required in `" + componentName + "`, but its value is `" + propValue + "`.";
    },
    invalid: function (propName, propValue, componentName, validationName) {
        return "Invalid prop `" + propName + "` of type `" + typeof propValue + "` supplied to `" + componentName + "`, expected `" + validationName + "`";
    },
};
/**
 * Creates a prop type generator function based on the provided options.
 */
function createPropType(options) {
    var validationName = options.validationName, predicate = options.predicate, customWarnings = options.warnings;
    var warnings = Object.assign({}, defaultWarnings, customWarnings);
    invariant_1(predicate, 'Invalid "predicate" option supplied to custom prop type creator. ' + 'Expected a function, but got: %s', predicate);
    function generateValidator(isRequired) {
        if (isRequired === void 0) { isRequired = false; }
        return function (props, propName, componentName) {
            var propValue = props[propName];
            if (typeof propValue === 'undefined' || propValue === null) {
                invariant_1(!isRequired, warnings.missing(propName, propValue, componentName));
                return;
            }
            invariant_1(predicate(propValue), warnings.invalid(propName, propValue, componentName, validationName));
            return;
        };
    }
    var propType = generateValidator();
    propType.isRequired = generateValidator(true);
    return propType;
}

function isColor(prop) {
    return (
    // hex
    /^#(?:[A-Fa-f0-9]{3}){1,2}$/i.test(prop) ||
        // rgb
        /^rgb[(](?:\s*0*(?:\d\d?(?:\.\d+)?(?:\s*%)?|\.\d+\s*%|100(?:\.0*)?\s*%|(?:1\d\d|2[0-4]\d|25[0-5])(?:\.\d+)?)\s*(?:,(?![)])|(?=[)]))){3}[)]$/i.test(prop) ||
        // rgba
        /^^rgba[(](?:\s*0*(?:\d\d?(?:\.\d+)?(?:\s*%)?|\.\d+\s*%|100(?:\.0*)?\s*%|(?:1\d\d|2[0-4]\d|25[0-5])(?:\.\d+)?)\s*,){3}\s*0*(?:\.\d+|1(?:\.0*)?)\s*[)]$/i.test(prop) ||
        // hsl
        /^hsl[(]\s*0*(?:[12]?\d{1,2}|3(?:[0-5]\d|60))\s*(?:\s*,\s*0*(?:\d\d?(?:\.\d+)?\s*%|\.\d+\s*%|100(?:\.0*)?\s*%)){2}\s*[)]$/i.test(prop) ||
        // hsla
        /^hsla[(]\s*0*(?:[12]?\d{1,2}|3(?:[0-5]\d|60))\s*(?:\s*,\s*0*(?:\d\d?(?:\.\d+)?\s*%|\.\d+\s*%|100(?:\.0*)?\s*%)){2}\s*,\s*0*(?:\.\d+|1(?:\.0*)?)\s*[)]$/i.test(prop));
}

var colorValidator = createPropType({ validationName: 'color', predicate: function (propValue) { return isColor(propValue); } });
var color = colorValidator;

exports.color = color;
//# sourceMappingURL=index.js.map
