import { Button, Grid } from '@material-ui/core';
import React from 'react';
import CustomDatePicker from './CustomDatePicker';
import ViewContainer from './ViewContainer';
import CustomTable from './CustomTable';
import { CreditCard } from '@material-ui/icons';
import CustomInputField from './CustomInputField';
import { API } from '../config';
import ActivityIndicator from './ActivityIndicator';
import { Skeleton } from '@material-ui/lab';
const moment = require('moment');

const columns = [
  { id: 'Trans_Name', label: 'Name', minWidth: 50 },
  { id: 'Trans_Mobile', label: 'Mobile', minWidth: 100 },
  {
    id: 'Trans_Amount',
    label: 'Amount',
    minWidth: 100,
    format: (value) => value.toLocaleString('en-US'),
  },
  {
    id: 'Date',
    label: 'Date',
    minWidth: 100,
    format: (value) => value.toLocaleString('en-US'),
  },
  {
    id: 'Trans_Mpesa_No',
    label: 'Mpesa\u00a0Code',
    minWidth: 100,
    format: (value) => value.toLocaleString('en-US'),
  },
  {
    id: 'Trans_Type',
    label: 'Type',
    minWidth: 100,
    format: (value) => value.toLocaleString('en-US'),
  },
  {
    id: 'Trans_Status',
    label: 'Status',
    minWidth: 100,
    format: (value) => value.toLocaleString('en-US'),
  },
];

function Payments(props) {
  const today = moment().format('YYYY-MM-DD');

  const [selectedFromDate, setSelectedFromDate] = React.useState(
    moment(today).format('YYYY-MM-DD')
  );
  const [selectedToDate, setSelectedToDate] = React.useState(
    moment(today).format('YYYY-MM-DD')
  );

  const [rows, setRows] = React.useState([]);
  const [loading, setLoading] = React.useState(false);
  const [mounting, setMounting] = React.useState(false);
  const [refresh, setRefresh] = React.useState(false);
  const [phoneNumber, setPhoneNumber] = React.useState('');
  console.log(phoneNumber);

  React.useEffect(() => {
    const fetchData = async () => {
      setMounting(true);
      const res = await fetch(`${API}/api/payments`);
      const json = await res.json();
      setRows(json);
      setMounting(false);
    };
    fetchData();
  }, [refresh]);

  //Filter records
  const filterData = async () => {
    setLoading(true);
    const res = await fetch(`${API}/api/filter/payments`, {
      method: 'POST', // or 'PUT'
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({ From: selectedFromDate, To: selectedToDate }),
    });
    const json = await res.json();
    setRows(json);
    setLoading(false);
  };

  const searchDataByMobile = async (phoneNumber) => {
    setLoading(true);
    const res = await fetch(`${API}/api/search/payments`, {
      method: 'POST', // or 'PUT'
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({ mobile: phoneNumber }),
    });
    const json = await res.json();
    setRows(json);
    setLoading(false);
  };

  const handleFromDateChange = (date) => {
    setSelectedFromDate(date);
  };

  const handleInputFieldChange = (e) => {
    setPhoneNumber(e.target.value);
  };

  const handleToDateChange = (date) => {
    setSelectedToDate(date);
  };

  const searchDataOnEnter = (event) => {
    if (event.key === 'Enter') {
      // Do code here
      event.preventDefault();
      if (phoneNumber.length >= 10) {
        searchDataByMobile(phoneNumber);
      }
    }
  };

  return (
    <>
      <ActivityIndicator visible={loading} />
      <ViewContainer icon={<CreditCard />} title="Payments">
        {mounting ? (
          <>
            <Skeleton variant="text" />
            <Skeleton variant="text" />
            <Skeleton variant="text" />
            <Skeleton variant="text" />
            <Skeleton variant="text" />

            <Skeleton
              variant="rect"
              width="100%"
              height={518}
              animation="wave"
            />
          </>
        ) : (
          <>
            <Grid
              container
              spacing={2}
              style={{ display: 'flex', justifyContent: 'flex-end' }}
            >
              <Grid item>
                <CustomInputField
                  label="Search By Mobile"
                  variant="outlined"
                  size="small"
                  placeholder="Enter mobile number"
                  style={{ top: '1rem' }}
                  onChange={handleInputFieldChange}
                  onKeyDown={searchDataOnEnter}
                />
              </Grid>
              <Grid item>
                <CustomDatePicker
                  selectedDate={selectedFromDate}
                  handleDateChange={handleFromDateChange}
                  label="Select Start Date"
                />
              </Grid>
              <Grid item>
                <CustomDatePicker
                  selectedDate={selectedToDate}
                  handleDateChange={handleToDateChange}
                  label="Select End Date"
                />
              </Grid>
              <Grid item>
                <Button
                  variant="contained"
                  color="primary"
                  style={{ top: '1rem', width: 100 }}
                  size="medium"
                  onClick={() => setRefresh(!refresh)}
                >
                  Refresh
                </Button>
              </Grid>
              <Grid item>
                <Button
                  variant="contained"
                  color="primary"
                  style={{ top: '1rem', width: 100 }}
                  size="medium"
                  onClick={filterData}
                >
                  Filter
                </Button>
              </Grid>
            </Grid>
            <CustomTable rows={rows} columns={columns} />
          </>
        )}
      </ViewContainer>
    </>
  );
}

export default Payments;
