import { Button, Grid, TableCell, TableRow } from '@material-ui/core';
import React from 'react';
import CustomInputField from './CustomInputField';
import ViewContainer from './ViewContainer';
import CustomTable from './CustomTable';
import { ReceiptRounded } from '@material-ui/icons';
import CustomDropDown from './CustomDropDown';
import { API } from '../config';
import ActivityIndicator from './ActivityIndicator';
import { isAuthenticated } from '../API';
import CustomDateTimePicker from './CustomDateTimePicker';
const moment = require('moment');

const columns = [
  { id: 'Trans_Name', label: 'Name', minWidth: 250 },
  { id: 'Trans_Mobile', label: 'Mobile', minWidth: 100 },
  {
    id: 'Date',
    label: 'Date',
    minWidth: 200,
    format: (value) => value.toLocaleString('en-US'),
  },
  {
    id: 'Trans_Mpesa_No',
    label: 'Mpesa\u00a0Code',
    minWidth: 100,
    format: (value) => value.toLocaleString('en-US'),
  },
  {
    id: 'Trans_Acc_No',
    label: 'Account\u00a0Number',
    minWidth: 100,
    format: (value) => value.toLocaleString('en-US'),
  },
  {
    id: 'Trans_Amount',
    label: 'Points',
    minWidth: 100,
    format: (value) => value.toLocaleString('en-US'),
  },
  {
    id: 'Trans_Amount',
    label: 'Amount',
    minWidth: 100,
    format: (value) => value.toLocaleString('en-US'),
  },
];

function Transactions(props) {
  const today = moment().format('YYYY-MM-DD HH:mm');
  const { mobile } = isAuthenticated();

  const [selectedFromDate, setSelectedFromDate] = React.useState(
    moment(today).format('YYYY-MM-DD HH:mm')
  );
  const [selectedToDate, setSelectedToDate] = React.useState(
    moment(today).format('YYYY-MM-DD HH:mm')
  );

  console.log(`From Date: ${selectedFromDate}`);
  console.log(`To Date: ${selectedToDate}`);

  const [rows, setRows] = React.useState([]);
  const [loading, setLoading] = React.useState(false);
  const [refresh, setRefresh] = React.useState(false);
  const [companyIDMapping, setCompanyIDMapping] = React.useState();
  const [filterMessagePlatform, setFilterMessagePlatform] = React.useState(1);
  const [phoneNumber, setPhoneNumber] = React.useState('');
  const [chosenCompanyMpesaTable, setChosenComapnyMpesaTable] = React.useState(
    mobile === '254720740895'
      ? 'limitless_transactions_c2b'
      : mobile === '254708672888'
      ? 'own_your_game_transactions_c2b'
      : mobile === '254711664398'
      ? 'jibambe_transactions_c2b'
      : 'dairy_transactions_c2b'
  );
  const [mounting, setMounting] = React.useState(false);

  React.useEffect(() => {
    const fetchData = async () => {
      setMounting(true);
      const res = await fetch(`${API}/api/company-ID-mappings`, {
        method: 'POST', // or 'PUT'
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({ mobile }),
      });
      const json = await res.json();
      setCompanyIDMapping(json);
      setMounting(false);
    };
    fetchData();
  }, []);

  React.useEffect(() => {
    const fetchData = async () => {
      setLoading(true);
      const res = await fetch(`${API}/api/transactions`, {
        method: 'POST', // or 'PUT'
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({ mobile, prodNumber: '37' }),
      });
      const json = await res.json();
      setRows(json);
      setLoading(false);
    };
    fetchData();
  }, [refresh]);

  const sentMessageFilterByPlatform = async (e) => {
    console.log(e.target.value);
    setFilterMessagePlatform(e.target.value);
    const res1 = await fetch(`${API}/api/mpesa-table-name-by-product-number`, {
      method: 'POST', // or 'PUT'
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({ productNo: e.target.value }),
    });
    const json1 = await res1.json();
    setChosenComapnyMpesaTable(json1[0].Mpesa_Table_Name);

    console.log(chosenCompanyMpesaTable);
    setLoading(true);
    const res = await fetch(`${API}/api/filter/transactions-by-platform`, {
      method: 'POST', // or 'PUT'
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({
        table: json1[0].Mpesa_Table_Name,
        mobile,
      }),
    });
    const json = await res.json();
    setRows(json);
    setLoading(false);
  };

  const searchDataByMobileAndPlatform = async (phoneNumber, table) => {
    setLoading(true);
    const res = await fetch(
      `${API}/api/search/transactions-by-platform-and-mobile`,
      {
        method: 'POST', // or 'PUT'
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({ mobile: phoneNumber, table: table }),
      }
    );
    const json = await res.json();
    setRows(json);
    setLoading(false);
  };

  //Filter records
  const filterDataByPlatformAndDate = async () => {
    setLoading(true);
    const res = await fetch(
      `${API}/api/filter/transactions-by-platform-and-date`,
      {
        method: 'POST', // or 'PUT'
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          table: chosenCompanyMpesaTable,
          From: moment(selectedFromDate).format('YYYY-MM-DD HH:mm'),
          To: moment(selectedToDate).format('YYYY-MM-DD HH:mm'),
          mobile,
        }),
      }
    );
    const json = await res.json();
    setRows(json);
    setLoading(false);
  };

  const handleFromDateChange = (date) => {
    setSelectedFromDate(date);
  };

  const handleInputFieldChange = (e) => {
    setPhoneNumber(e.target.value);
  };

  const handleToDateChange = (date) => {
    setSelectedToDate(date);
  };

  const searchDataOnEnter = (event) => {
    if (event.key === 'Enter') {
      // Do code here
      console.log(chosenCompanyMpesaTable);
      event.preventDefault();
      if (phoneNumber.length >= 10) {
        searchDataByMobileAndPlatform(phoneNumber, chosenCompanyMpesaTable);
      }
    }
  };

  function sumAmount(items) {
    return items
      .map(({ Trans_Amount }) => Trans_Amount)
      .reduce((count, i) => count + parseInt(i), 0);
  }

  const summaryHeaderData = () => {
    return (
      <TableRow style={{ backgroundColor: '#3C8016' }}>
        <TableCell />
        <TableCell />
        <TableCell />
        <TableCell />
        <TableCell />
        <TableCell
          style={{
            fontSize: '10px',
            fontWeight: 'bolder',
            color: 'whitesmoke',
          }}
          colSpan={1}
        >
          Total Transactions {parseInt(rows.length.toLocaleString())}
        </TableCell>
        <TableCell
          style={{
            fontSize: '10px',
            fontWeight: 'bolder',
            color: 'whitesmoke',
          }}
          align="left"
        >
          Total Amount {parseInt(sumAmount(rows)).toLocaleString()}
        </TableCell>
      </TableRow>
    );
  };

  return (
    <>
      <ActivityIndicator visible={loading} />
      <ViewContainer icon={<ReceiptRounded />} title="Transactions">
        <Grid
          container
          spacing={2}
          style={{ display: 'flex', justifyContent: 'flex-end' }}
        >
          <Grid item>
            <CustomDropDown
              variant="outlined"
              size="small"
              // label="Select PlatForm"
              style={{ marginRight: '10px', width: '10rem' }}
              selectedValue={filterMessagePlatform}
              handleChange={sentMessageFilterByPlatform}
              data={companyIDMapping}
            />
          </Grid>
          <Grid item>
            <CustomInputField
              label="Search By Mobile"
              variant="outlined"
              size="small"
              placeholder="Enter mobile number"
              onChange={handleInputFieldChange}
              onKeyDown={searchDataOnEnter}
            />
          </Grid>
          <Grid item>
            <CustomDateTimePicker
              selectedDate={selectedFromDate}
              handleDateTimeChange={handleFromDateChange}
              helperText="Select Start Date"
            />
          </Grid>
          <Grid item>
            <CustomDateTimePicker
              selectedDate={selectedToDate}
              handleDateTimeChange={handleToDateChange}
              helperText="Select End Date"
            />
          </Grid>
          <Grid item>
            <Button
              variant="contained"
              color="primary"
              style={{ bottom: '1rem', width: 100 }}
              size="medium"
              onClick={() => setRefresh(!refresh)}
            >
              Refresh
            </Button>
          </Grid>
          <Grid item>
            <Button
              variant="contained"
              color="primary"
              style={{ bottom: '1rem', width: 100 }}
              size="medium"
              onClick={filterDataByPlatformAndDate}
            >
              Filter
            </Button>
          </Grid>
        </Grid>
        <CustomTable
          rows={rows}
          columns={columns}
          summaryHeader={true}
          summaryHeaderData={summaryHeaderData()}
        />
      </ViewContainer>
    </>
  );
}

export default Transactions;
