import {
  Avatar,
  Button,
  makeStyles,
  Modal,
  Paper,
  TextField,
  Typography,
} from '@material-ui/core';
import { GroupWorkRounded } from '@material-ui/icons';
import { Skeleton } from '@material-ui/lab';
import React, { useState } from 'react';
import ViewContainer from './ViewContainer';
import { API } from '../config';
import { isAuthenticated } from '../API';

const useStyles = makeStyles((theme) => ({
  root: {
    display: 'flex',
    flexDirection: 'column',
    alignItems: 'center',
    padding: theme.spacing(3),
  },
  avatar: {
    width: theme.spacing(10),
    height: theme.spacing(10),
    marginBottom: theme.spacing(2),
  },
  userInfo: {
    marginBottom: theme.spacing(2),
  },
  button: {
    marginTop: theme.spacing(2),
  },
  modal: {
    display: 'flex',
    alignItems: 'center',
    justifyContent: 'center',
  },
  modalContent: {
    backgroundColor: theme.palette.background.paper,
    boxShadow: theme.shadows[5],
    padding: theme.spacing(4),
    outline: 'none',
  },
  textField: {
    marginBottom: theme.spacing(2),
  },
}));

function UserProfile(props) {
  const { ID } = isAuthenticated();
  const classes = useStyles();
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [newFirstName, setNewFirstName] = useState('');
  const [newLastName, setNewLastName] = useState('');
  const [firstName, setFirstName] = useState('');
  const [lastName, setLastName] = useState('');
  const [email, setEmail] = useState('');
  const [newProfilePicture, setNewProfilePicture] = useState(null);
  const [selectedProfilePicture, setSelectedProfilePicture] = useState(null);
  const [mounting, setMounting] = React.useState(false);

  React.useEffect(() => {
    async function getData() {
      const response = await fetch(`${API}/api/get-profile-data`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({ id: ID }),
      });

      if (response.ok) {
        const data = await response.json();
        // Clock added successfully, fetch the updated list of clocks
        console.log('ok', data);
        const cleanedData = data[0];
        setNewFirstName(cleanedData.Firstname);
        setNewLastName(cleanedData.Lastname);
        setEmail(cleanedData.Email);
        setLastName(cleanedData.Lastname);
        setFirstName(cleanedData.Firstname);
        setNewProfilePicture(cleanedData.Profile_url);
      } else {
        // Handle error case
        console.log(response);
        throw new Error('Error adding clock');
      }
    }

    getData();
  }, []);

  React.useEffect(() => {
    setMounting(true);
    setMounting(false);
  }, []);

  const handleEditProfile = () => {
    setIsModalOpen(true);
  };

  const handleSaveChanges = () => {
    // Handle save changes logic here
    let formData = new FormData();

    if (selectedProfilePicture) {
      formData.append('file', selectedProfilePicture);
    }

    formData.append('userID', ID);
    formData.append('lastName', newLastName);
    formData.append('firstName', newFirstName);

    fetch(`${API}/api/update-profile-data`, {
      method: 'POST',
      body: formData,
    })
      .then((response) => {
        if (response.ok) {
          // Successfully updated profile data
          console.log('Profile data updated');
          window.location.reload();
        } else {
          // Handle error case
          console.error('Error updating profile data');
        }
      })
      .catch((error) => {
        // Handle error case
        console.error('Error updating profile data:', error);
      });

    setIsModalOpen(false);
    // Update profile with new values
  };

  const handleCancelChanges = () => {
    setIsModalOpen(false);
    // Reset input fields if needed
  };

  const handleFirstNameChange = (event) => {
    setNewFirstName(event.target.value);
  };

  const handleLastNameChange = (event) => {
    setNewLastName(event.target.value);
  };

  const handleProfilePictureChange = (event) => {
    const file = event.target.files[0];
    setSelectedProfilePicture(file);
    // Check if FileReader is available (client-side)
    if (typeof FileReader !== 'undefined') {
      const reader = new FileReader();
      reader.onload = () => {
        setNewProfilePicture(reader.result);
      };
      reader.readAsDataURL(file);
    }
  };

  return (
    <ViewContainer icon={<GroupWorkRounded />} title="Users">
      {mounting ? (
        <>
          <Skeleton variant="text" />
          <Skeleton variant="text" />
          <Skeleton variant="text" />
          <Skeleton variant="text" />
          <Skeleton variant="text" />

          <Skeleton variant="rect" width="100%" height={518} animation="wave" />
        </>
      ) : (
        <div>
          <Paper className={classes.root}>
            <Avatar
              alt="Profile Picture"
              src={newProfilePicture}
              className={classes.avatar}
            />
            <Typography variant="h5" className={classes.userInfo}>
              {`${firstName} ${lastName}`}
            </Typography>
            <Typography variant="body1" className={classes.userInfo}>
              Email: {email}
            </Typography>
            <Button
              variant="contained"
              color="primary"
              className={classes.button}
              onClick={handleEditProfile}
            >
              Edit Profile
            </Button>
          </Paper>

          <Modal
            open={isModalOpen}
            onClose={handleCancelChanges}
            className={classes.modal}
          >
            <div className={classes.modalContent}>
              <div style={{ display: 'flex', justifyContent: 'space-between' }}>
                <Typography variant="h6" gutterBottom>
                  Edit Profile
                </Typography>
                <div
                  style={{
                    height: 40,
                    width: 40,
                    background: 'grey',
                    borderRadius: '50%',
                    display: 'flex',
                    justifyContent: 'center',
                    alignItems: 'center',
                    cursor: 'pointer',
                  }}
                  onClick={handleCancelChanges}
                >
                  X
                </div>
              </div>
              <label htmlFor="profilePictureInput">
                <div className="profile-photo-upload">
                  <label htmlFor="file-input">
                    {newProfilePicture ? (
                      <img src={newProfilePicture} alt="Preview" />
                    ) : (
                      <div className="upload-placeholder">
                        <span>+</span>
                        <p>Upload a photo</p>
                      </div>
                    )}
                  </label>
                  <input
                    id="file-input"
                    name="file"
                    type="file"
                    accept="image/*"
                    onChange={handleProfilePictureChange}
                  />
                </div>
              </label>
              <TextField
                label="First Name"
                value={newFirstName}
                onChange={handleFirstNameChange}
                className={classes.textField}
                fullWidth
              />
              <TextField
                label="Last Name"
                value={newLastName}
                onChange={handleLastNameChange}
                className={classes.textField}
                fullWidth
              />
              <div style={{ display: 'flex', justifyContent: 'space-between' }}>
                <Button
                  variant="contained"
                  color="primary"
                  onClick={handleSaveChanges}
                >
                  Save Changes
                </Button>
                <Button
                  variant="contained"
                  color="secondary"
                  onClick={handleCancelChanges}
                >
                  Cancel
                </Button>
              </div>
            </div>
          </Modal>
        </div>
      )}
    </ViewContainer>
  );
}

export default UserProfile;
