import React, { useEffect, useState } from 'react';
import MyClock from './MyClock';
import timezone from './timezone.json';
import Autocomplete from '@material-ui/lab/Autocomplete';
import TextField from '@material-ui/core/TextField';
import ViewContainer from './ViewContainer';
import { AccessTimeOutlined } from '@material-ui/icons';
import { API } from '../config';

const WorldCenter = () => {
  const [clocks, setClocks] = useState([]);
  const [selectedClock, setSelectedClock] = useState('Africa/Nairobi');
  const [country, setCountry] = useState('Kenya');
  const [offset, setOffset] = useState('180');
  const [iso, setIso] = useState('KE');
  const [refresh, setRefresh] = useState(false);

  useEffect(() => {
    fetch(`${API}/api/get-clocks`)
      .then((response) => {
        if (response.ok) {
          return response.json();
        } else {
          // Handle error case
          throw new Error('Error fetching clocks');
        }
      })
      .then((data) => {
        setClocks(data);
      })
      .catch((error) => {
        // Handle error case
      });
  }, []);

  const handleChange = (e, value) => {
    if (value) {
      setSelectedClock(value.Timezone);
      setCountry(value.Country);
      setOffset(value.Offset);
      setIso(value.ISO);
    }
  };

  const fetchClocks = async () => {
    try {
      const response = await fetch(`${API}/api/get-clocks`);

      if (response.ok) {
        const data = await response.json();
        setClocks(data); // Update setClocks with the fetched data
        setRefresh(!refresh);
      } else {
        // Handle error case
        throw new Error('Error fetching clocks');
      }
    } catch (error) {
      // Handle error case
    }
  };

  const addTimeZone = async (event) => {
    event.preventDefault(); // Prevent form submission behavior

    if (clocks.findIndex((c) => c.Timezone === selectedClock) < 0) {
      const zone = {
        Timezone: selectedClock,
        Country: country,
        Offset: offset,
        ISO: iso,
      };

      try {
        const response = await fetch(`${API}/api/add-clock`, {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
          },
          body: JSON.stringify(zone),
        });

        if (response.ok) {
          // Clock added successfully, fetch the updated list of clocks
          fetchClocks();
        } else {
          // Handle error case
          throw new Error('Error adding clock');
        }
      } catch (error) {
        // Handle error case
      }

      // Reset form inputs
      setSelectedClock(null);
      setCountry('');
      setOffset('');
    }
  };

  const removeClock = (id) => {
    const updatedClocks = clocks.filter((c) => c.id !== id);
    setClocks(updatedClocks);
  };

  let clockItems = null;
  if (clocks && clocks.length > 0) {
    clockItems = clocks.map((zone, index) => (
      <MyClock
        {...zone}
        key={`${zone.Timezone}-${index}`}
        onRemove={removeClock}
      />
    ));
  }

  return (
    <ViewContainer icon={<AccessTimeOutlined />} title="World Clock">
      <div className="w3-container">
        <div style={{ display: 'flex', alignItems: 'center' }}>
          <Autocomplete
            options={timezone}
            value={timezone.find((option) => option.Timezone === selectedClock)}
            onChange={handleChange}
            getOptionLabel={(option) =>
              `${option.Country} - ${option.Timezone}`
            }
            renderInput={(params) => (
              <TextField
                {...params}
                label="Select a timezone"
                variant="outlined"
                style={{ minWidth: '15rem', marginRight: '1rem' }}
              />
            )}
            style={{ flex: '1' }}
          />
          <button
            onClick={addTimeZone}
            style={{
              backgroundColor: '#1976d2',
              color: 'white',
              border: 'none',
              borderRadius: '4px',
              padding: '10px 20px',
              fontWeight: 'bold',
              cursor: 'pointer',
              transition: 'background-color 0.3s ease-in-out',
            }}
          >
            ADD CLOCK
          </button>
        </div>
        <div
          style={{
            display: 'flex',
            flexWrap: 'wrap',
            marginTop: '1rem',
          }}
        >
          {clockItems}
        </div>
      </div>
    </ViewContainer>
  );
};

export default WorldCenter;
